// Admin Simple Dashboard JavaScript

// Load customers when page loads
document.addEventListener('DOMContentLoaded', function() {
    loadCustomers();
    setupEventListeners();
});

// Setup event listeners
function setupEventListeners() {
    // Add customer form
    document.getElementById('add-customer-form').addEventListener('submit', addCustomer);
    
    // Search input
    document.getElementById('search-input').addEventListener('input', filterCustomers);
    
    // Mobile number validation
    document.getElementById('mobile').addEventListener('input', function(e) {
        e.target.value = e.target.value.replace(/[^0-9]/g, '').substring(0, 10);
    });
    
    document.getElementById('edit-mobile').addEventListener('input', function(e) {
        e.target.value = e.target.value.replace(/[^0-9]/g, '').substring(0, 10);
    });
}

// Load all customers
function loadCustomers() {
    fetch('fetchRecords.php')
        .then(response => response.json())
        .then(data => {
            console.log('Loaded customers:', data);
            displayCustomers(data);
            updateStats(data);
        })
        .catch(error => {
            console.error('Error loading customers:', error);
            showToast('Failed to load customers', 'error');
            document.getElementById('customers-tbody').innerHTML = `
                <tr class="empty-state">
                    <td colspan="6">
                        <i class="fas fa-exclamation-triangle"></i>
                        <h3>Error Loading Data</h3>
                        <p>Failed to load customer records. Please refresh the page.</p>
                    </td>
                </tr>
            `;
        });
}

// Display customers in table
function displayCustomers(customers) {
    const tbody = document.getElementById('customers-tbody');
    
    if (!customers || customers.length === 0) {
        tbody.innerHTML = `
            <tr class="empty-state">
                <td colspan="6">
                    <i class="fas fa-users"></i>
                    <h3>No Customers Found</h3>
                    <p>Start by adding your first customer above.</p>
                </td>
            </tr>
        `;
        return;
    }
    
    tbody.innerHTML = customers.map(customer => `
        <tr data-customer-id="${customer.id}">
            <td><strong>${customer.name}</strong></td>
            <td>${customer.mobile}</td>
            <td>${customer.email || '-'}</td>
            <td>${customer.policy_number || '-'}</td>
            <td>${formatDate(customer.created_at)}</td>
            <td>
                <div class="table-actions">
                    <button class="btn btn-sm btn-secondary" onclick="editCustomer(${customer.id})">
                        <i class="fas fa-edit"></i> Edit
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="deleteCustomer(${customer.id}, '${customer.name}')">
                        <i class="fas fa-trash"></i> Delete
                    </button>
                </div>
            </td>
        </tr>
    `).join('');
}

// Add new customer
function addCustomer(e) {
    e.preventDefault();
    
    const form = e.target;
    const formData = new FormData(form);
    
    // Validate mobile number
    const mobile = formData.get('mobile');
    if (mobile.length !== 10) {
        showToast('Please enter a valid 10-digit mobile number', 'error');
        return;
    }
    
    fetch('addRecord.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log('Add customer response:', data);
        if (data.success) {
            showToast('Customer added successfully!', 'success');
            form.reset();
            loadCustomers();
        } else {
            showToast(data.error || 'Failed to add customer', 'error');
        }
    })
    .catch(error => {
        console.error('Error adding customer:', error);
        showToast('Failed to add customer', 'error');
    });
}

// Edit customer
let currentEditId = null;

function editCustomer(id) {
    fetch('fetchRecords.php')
        .then(response => response.json())
        .then(data => {
            const customer = data.find(c => c.id == id);
            if (customer) {
                currentEditId = id;
                document.getElementById('edit-id').value = id;
                document.getElementById('edit-name').value = customer.name;
                document.getElementById('edit-mobile').value = customer.mobile;
                document.getElementById('edit-email').value = customer.email || '';
                openEditModal();
            }
        })
        .catch(error => {
            console.error('Error loading customer:', error);
            showToast('Failed to load customer data', 'error');
        });
}

function saveEdit() {
    const id = document.getElementById('edit-id').value;
    const name = document.getElementById('edit-name').value;
    const mobile = document.getElementById('edit-mobile').value;
    const email = document.getElementById('edit-email').value;
    
    // Validate
    if (!name || !mobile) {
        showToast('Please fill all required fields', 'error');
        return;
    }
    
    if (mobile.length !== 10) {
        showToast('Please enter a valid 10-digit mobile number', 'error');
        return;
    }
    
    const formData = new FormData();
    formData.append('id', id);
    formData.append('name', name);
    formData.append('mobile', mobile);
    formData.append('email', email);
    
    fetch('updateRecord.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log('Update response:', data);
        if (data.success) {
            showToast('Customer updated successfully!', 'success');
            closeEditModal();
            loadCustomers();
        } else {
            showToast(data.error || 'Failed to update customer', 'error');
        }
    })
    .catch(error => {
        console.error('Error updating customer:', error);
        showToast('Failed to update customer', 'error');
    });
}

// Delete customer
let currentDeleteId = null;

function deleteCustomer(id, name) {
    currentDeleteId = id;
    document.getElementById('delete-customer-name').textContent = name;
    openDeleteModal();
}

function confirmDelete() {
    if (!currentDeleteId) return;
    
    const formData = new FormData();
    formData.append('id', currentDeleteId);
    
    fetch('deleteRecord.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log('Delete response:', data);
        if (data.success) {
            showToast('Customer deleted successfully!', 'success');
            closeDeleteModal();
            loadCustomers();
        } else {
            showToast(data.error || 'Failed to delete customer', 'error');
        }
    })
    .catch(error => {
        console.error('Error deleting customer:', error);
        showToast('Failed to delete customer', 'error');
    });
}

// Delete all records
function deleteAllRecords() {
    if (!confirm('⚠️ WARNING: This will delete ALL customer records!\n\nThis action cannot be undone. Are you absolutely sure?')) {
        return;
    }
    
    if (!confirm('This is your final confirmation. Delete ALL records?')) {
        return;
    }
    
    fetch('deleteAllRecords.php', {
        method: 'POST'
    })
    .then(response => response.json())
    .then(data => {
        console.log('Delete all response:', data);
        if (data.success) {
            showToast('All records deleted successfully!', 'success');
            loadCustomers();
        } else {
            showToast(data.error || 'Failed to delete records', 'error');
        }
    })
    .catch(error => {
        console.error('Error deleting all records:', error);
        showToast('Failed to delete records', 'error');
    });
}

// Filter customers (search)
let allCustomersData = [];

function filterCustomers() {
    const searchTerm = document.getElementById('search-input').value.toLowerCase();
    
    fetch('fetchRecords.php')
        .then(response => response.json())
        .then(data => {
            allCustomersData = data;
            
            if (!searchTerm) {
                displayCustomers(data);
                return;
            }
            
            const filtered = data.filter(customer => 
                customer.name.toLowerCase().includes(searchTerm) ||
                customer.mobile.includes(searchTerm) ||
                (customer.email && customer.email.toLowerCase().includes(searchTerm)) ||
                (customer.policy_number && customer.policy_number.toLowerCase().includes(searchTerm))
            );
            
            displayCustomers(filtered);
        })
        .catch(error => {
            console.error('Error filtering customers:', error);
        });
}

// Update stats
function updateStats(customers) {
    // Total customers
    document.getElementById('total-customers').textContent = customers.length;
    
    // New today
    const today = new Date().toISOString().split('T')[0];
    const newToday = customers.filter(c => c.created_at && c.created_at.startsWith(today)).length;
    document.getElementById('new-today').textContent = newToday;
    
    // Active policies (for now, same as total)
    document.getElementById('active-policies').textContent = customers.length;
}

// Modal functions
function openEditModal() {
    document.getElementById('edit-modal').classList.add('active');
}

function closeEditModal() {
    document.getElementById('edit-modal').classList.remove('active');
    currentEditId = null;
}

function openDeleteModal() {
    document.getElementById('delete-modal').classList.add('active');
}

function closeDeleteModal() {
    document.getElementById('delete-modal').classList.remove('active');
    currentDeleteId = null;
}

// Toast notification
function showToast(message, type = 'success') {
    const toast = document.getElementById('toast');
    
    const icon = type === 'success' ? 'fa-check-circle' : 
                 type === 'error' ? 'fa-exclamation-circle' : 
                 'fa-info-circle';
    
    toast.innerHTML = `
        <div class="toast-icon">
            <i class="fas ${icon}"></i>
        </div>
        <div class="toast-message">${message}</div>
    `;
    
    toast.className = `toast ${type} show`;
    
    setTimeout(() => {
        toast.classList.remove('show');
    }, 4000);
}

// Utility functions
function formatDate(dateString) {
    if (!dateString) return '-';
    const date = new Date(dateString);
    return date.toLocaleDateString('en-US', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

// Logout function
function logout() {
    if (confirm('Are you sure you want to logout?')) {
        window.location.href = 'login.html';
    }
}

// Close modals when clicking outside
window.addEventListener('click', function(e) {
    if (e.target.classList.contains('modal')) {
        closeEditModal();
        closeDeleteModal();
    }
});

// Close modals with Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        closeEditModal();
        closeDeleteModal();
    }
});
